package spacing_functions;

#  spacing_functions module
#
# simple functions used to calculate probability spacings
# for binary_grid/binary_grid2 (part of the binary_c project )
#

our $VERSION = '0.02';

# V 0.01 : initial attempt, ported most functions, new 'bastard' distribution 
#          to blend work of Sana et al 2012, Duchene and Kraus 2013, Raghavan 2010 etc.
#
# V 0.02 : use thread queue properly instead of old-fashioned grid split

use 5.008005;
use strict;
use warnings;
use Sort::Key qw(nsort);  
#use Memoize;
#memoize('const');
use Carp qw(confess);
use Data::Dumper;

# rob's modules
use rob_misc;
use binary_stars;
use Math::Trig;
use binary_grid2;
use RobInterpolation qw/generic_interpolation_wrapper/;
use threads;
use threads::shared;
use Thread::Queue;

# module variables
use vars qw( $vb );

# module subroutines
use Hash::RobMerge;
use Time::HiRes qw(sleep);

require Exporter;

our @ISA = qw(Exporter);

my @funcs = qw(const number);

our %EXPORT_TAGS = ( 'all' => [ @funcs ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{all} } );

our @EXPORT = @funcs;

# constants
use constant PI => acos(0.0)*2.0;
use constant GAUSSIAN_PREFACTOR => (1.0/sqrt(2.0*PI));
use constant logmin => 1e-30;
use constant value_of_log10 => log(10.0);
use constant log_ln_converter => (1.0/log(10.0));

my $vb=0; # debugging output
my %const_dt_results; # const dt data cache
my @const_dt_mass_array;
my %const_dt_mass_spacing;
my $const_dt_mass_array_initial_length;

sub const
{
    # a constant spacing function between $min and $max with $n steps
    # (assuming grid points located centrally in zones)
    return (($_[1]-$_[0])/(1.0*$_[2]));
}

sub number
{
    # simply return the number passed in, used as a dummy function
    return $_[0];
}

sub const_dt
{
    # a spacing function that tries to fix dtime rather than dmass
    my $opts=$_[0]; # pass options in in a hash
    my $function=$_[1]; # what to do?

    # override verbosity ?
    $vb = $$opts{vb}//$vb;

    print "spacing_functions:const_dt: called : opts=$opts, function=$function : const_dt_mass_array length = ",scalar @const_dt_mass_array," (was ",($const_dt_mass_array_initial_length//''),")\n"if($vb);

    # make required data if it hasn't been made or if the reset call is given
    if($#const_dt_mass_array==-1 || $function eq 'reset')
    { 
	print "spacing_functions:const_dt: called first time, or to reset\n"if($vb);

	return undef if($function ne 'reset' && defined($const_dt_mass_array_initial_length));

	no warnings;
	my $optsstring=join '',sort keys %$opts, sort values %$opts;
	use warnings;

        # TODO : FIXME
	#binary_grid::flexigrid_grid_resolution_shift() if($function eq 'reset');

	if(defined($const_dt_results{$optsstring}))
	{
	    # use cached values if available
	    print "spacing_functions:const_dt: Use cached masses array\n"if($vb);
	    @const_dt_mass_array=@{$const_dt_results{$optsstring}{mass_array}};
	    %const_dt_mass_spacing=%{$const_dt_results{$optsstring}{mass_hash}};
	    #print "spacing_functions:const_dt: const_dt_mass_array size now $#const_dt_mass_array\n"if($vb);
	}
	else
	{
	    print "spacing_functions:const_dt: No cached mass list available for current options\n"if($vb);
	    # space for results (to be filled)
	    my %results=(evoltimes_hash => {},
			 evoltimes_array => [],
			 mtimes_hash => {},
			 mtimes_array => [],
			 mass_hash => {},
			 mass_array => [],
			 initial_length => undef,
		);
	    if($vb)
	    {
		map
		{
		    print "spacing_functions:const_dt opt '$_' = '$$opts{$_}'\n" if(defined $_ && defined $$opts{$_});
		}sort keys %$opts;
		print "Make const_dt data opts=$opts results=",\%results,"\n";
	    }  
            
            #print "Call adaptive_masshash with opts \n",Data::Dumper->Dump([$opts]),"\n";

	    spacing_functions::adaptive_masshash($opts,\%results);
	    
	    # cache results
	    $const_dt_results{$optsstring} = \%results;
	    @const_dt_mass_array=@{$results{mass_array}}; # array which can be shifted
	    %const_dt_mass_spacing=%{$results{mass_hash}}; # array which can be shifted
	    $const_dt_mass_array_initial_length = scalar @const_dt_mass_array;

	    if($vb)
	    {
		print "Make const_dt_mass_array : ";
		map
		{
		    printf "[M=%g t=%g] ",$_,stellar_lifetime($_,\%results);
		}@const_dt_mass_array;
		print "\n";
	    }
	}
	if($function eq 'reset')
	{
	    print "spacing_functions:const_dt reset\n" if($vb);
	    return;
	}
    }

    # if $opts is just 'resolution' return the number of items in the array
    if($function eq 'resolution')
    {
	printf "spacing_functions:const_dt Resolution call == %d\n",$const_dt_mass_array_initial_length if($vb);
	return $const_dt_mass_array_initial_length;
    }
    elsif($function eq 'next')
    {
	# subsequent calls simply remove masses from the mass_array, or use the final
	# value if no more are available

	my $m = $#const_dt_mass_array==0 ? 
            $const_dt_mass_array[0] : shift @const_dt_mass_array;

	my $dlnm = $const_dt_mass_spacing{$m};

	print "spacing_functions:const_dt : Shift next mass off the list: M=$m > dlnM = $dlnm\n" if($vb);

	return $dlnm;
    }
    else
    {
	print STDERR "spacing_functions:const_dt called with unknown function '$function'\n";
	exit;
    }
}

sub adaptive_masshash
{
    # loop over time and make the masshash this way, 
    # retrospectively setting dlnm

    # input: 
    my $opts=$_[0]; # options hash
    
    # output:
    my $results=$_[1]; # results hash (see const_dt() for definition)

    my $defaults = {
        nthreads=>1,
        thread_sleep=>1,
        stellar_lifetime_table_nm=>1000,
        stellar_lifetime_cachedir=>undef,
        max_evolution_time=>13700.0,
        time_adaptive_mass_grid_log10_time=>0,
        time_adaptive_mass_grid_log10_step=>undef,
        extra_flash_resolution=>0,
        time_adapative_mass_grid_nlow_mass_stars=>10,
        max_delta_m=>10.0,
        vb=>0,
        savegrid=>0,
        debugging_output=>0,
        metallicity=>0.02,
        yields_dt=>1e8,
    };

    # opts required:
    # max_evolution_time 
    # stellar_lifetime_table_nm
    # nthreads
    # thread_sleep
    # time_adaptive_mass_grid_log10_time
    # time_adaptive_mass_grid_log10_step
    # time_adaptive_mass_grid_step
    # extra_flash_resolution
    # time_adaptive_mass_grid_nlow_mass_stars
    # max_delta_m
    # debugging_output
    # vb
    # savegrid
    # metallicity

    print "Apative masshash opts=$opts results=$results\n";
    if(defined $$opts{stellar_lifetime_cachedir})
    {
        print "Cache in $$opts{stellar_lifetime_cachedir}\n";
        rob_misc::mkdirhier($$opts{stellar_lifetime_cachedir});
    }
    
    # mlow and mhigh are hashes which are set for logging purposes only
    # if 'savegrid' is defined

    my $dt; # timestep
    my $dt0; # first timestep
    my $t0; # start time
    my $tmin; # minimum time
    my @masses;

    print "Called by ",(caller(2))[3],"\n";

    if($vb)
    {
	printf "adaptive_masshash : opts:\n";
	map
	{
	    printf "     $_ = ".(defined $$opts{$_} ? $$opts{$_} : 'undef')."\n" if(defined $_);
	}keys %$opts;
    }

    # check for a hash in the appropriate cahce directory
    print "Check cache for existing table... (in results = $results)\n";
    if(check_cache_for_stellar_lifetime_table($opts,$results)==0)
    {
        # nothing in the cache: make the data and save it
        # make stellar lifetime data table
        print "no existing table found, making a new one\n";
        make_stellar_lifetime_table($opts,$results);
        print "Made table\n";
    }
    else
    {
        print "Using table from cache\n";
    }
    
    # for the first two timesteps use a log grid to make
    # sure massive stars are resolved
    my $mbreak;

    # for masses < $mbreak we can fill the time bins as required
    my $lnmlow; # set below
    my $lnmhigh=log($$opts{mmax});

    my $logtimes=$$opts{time_adaptive_mass_grid_log10_time};

    # setup
    if($logtimes)
    {
	$logtimes=1;
	$dt=$$opts{time_adaptive_mass_grid_log10_step};
	print "t0 from dt=$dt, maxtime=$$opts{max_evolution_time}\n";
        $t0 = $dt * ( int( log10($$opts{max_evolution_time})/$dt ) - 0.5);
	$mbreak=100.0;
	$tmin = log10(stellar_lifetime($mbreak,$results)) - 1.0;
	$lnmlow = log(stellar_mass_from_lifetime(10.0**$t0, $results));
    }
    else
    {
	$logtimes=0;
	$dt=$$opts{time_adaptive_mass_grid_step};
        $t0=$dt*(int($$opts{max_evolution_time}/$dt)-0.5);
	$tmin=0.0;
	$mbreak=stellar_mass_from_lifetime($dt*2,$results);
	$lnmlow=log(stellar_mass_from_lifetime($t0,$results));
    }
    $dt0=$dt;
    
    print  "Time adaptive grid: dt=$dt, t0=$t0, tmin=$tmin, mbreak=$mbreak: lnmhigh=$lnmhigh (",exp($lnmhigh),")\n" if($vb);

    # set up helium flash data
    my $tflash;
    my $dtflash;
    my $mflash;
    if($$opts{extra_flash_resolution})
    {
	# calculate mass at which the helium flash happens
	$mflash=mass_helium_flash();
	
	# calculate time and duration of the helium flash
	$tflash=stellar_lifetime($mflash,$results);
	$dtflash = $logtimes ? log10(10.0**$tflash+10.0)*0.5 : 10.0; 
	
	# calculate nearest time bin
	$tflash=$dt*(int($tflash/$dt)+0.5);
	$tflash=log10($tflash) if($logtimes);
	printf "Extra He-flash resolution at t=%g +- %g\n",$tflash,$dtflash;
    }

    # include low-mass stars which would not make it
    # onto the main list (this is so that the grid probability adds to 1.0)
    #
    # Note that a *linear* grid is preferable if you want to resolve anything
    # in these low-mass stars, because the mass function is ~ flat at low mass
    
    if($$opts{time_adaptive_mass_grid_nlow_mass_stars}>0)
    {
	printf "Add %d low mass stars\n",$$opts{time_adaptive_mass_grid_nlow_mass_stars} if($vb);
	# force the stars to be in the very low mass range (<1Msun)
	# to make sure the probability adds to 1.0
	my $m_low=$$opts{mmin};
	my $m_high=1.0;
	
	my $m=$m_low;
	my $dm;
	push(@masses,$m);
	
	# can only loop if lower bound is really low mass
	if($m_high > $m_low+1e-4)
	{
	    $dm=($m_high-$m_low)/$$opts{time_adaptive_mass_grid_nlow_mass_stars};
	    while($m<=$m_high)
	    {
		push(@masses,$m);
		$m+=$dm;
	    }
	}
    }

    ###################################
    # make main part of the mass list #
    ###################################
        
    # we are required to start at a later time
    # than $t0 because there is contribution to (e.g) hertzsprung gap
    # stars at late times which is exaggerated because dlnm will be too
    # large... now we can assume these have ~20% of the MS lifetime (seems
    # to be ok...)
    
    my $tt0 = !$logtimes ? ($t0/0.8) : $t0;
    printf "Add normal stars from time %g to %g\n",($logtimes ? (10.0**$tmin,10.0**$tt0) : ($tmin,$tt0)) if($vb);
    
    for(my $t=$tt0;$t>$tmin;$t-=$dt)
    {	
	my $tt = $logtimes ? 10.0**$t : $t; # the time

	# from the lifetime, calculate the mass
	my $m = stellar_mass_from_lifetime($tt,$results);
	printf "From time %g (log10 = %g) > mass = %g (<mbreak=%g)\n",
            $tt,
            log10($tt),
            $m,
            $mbreak
            if($vb);
	
	last if ($m>=$mbreak);
	push(@masses,$m);
    }
    
    # replace bins at $tflash +/ $dt with higher resolution version
    if($$opts{extra_flash_resolution})
    {
	printf "Increase He flash resolution\n"if($vb);
	# increase flash resolution between $tmin and $tmax
	my $tmin=$tflash-$dtflash+0.5*$dt;
	my $tmax=$tflash+$dtflash-0.5*$dt;
	my $hedt=$dt*0.1;
	$tmin+=0.5*$hedt;
	$tmax-=0.5*$hedt;
	
	# calculate masses corresponding to $tmin and $tmax
	my $mmin=stellar_mass_from_lifetime($logtimes?10.0**$tmin:$tmin,$results);
	my $mmax=stellar_mass_from_lifetime($logtimes?10.0**$tmin:$tmin,$results);

	# and remove masses in that region from the masses array
	@masses=grep {$_<$mmin || $_>$mmax} @masses;
 	
	# now add extra flash resolution
	for(my $t=$tmin;$t<=$tmax;$t+=$hedt)
	{
	    push(@masses,stellar_mass_from_lifetime($logtimes?10.0**$t:$t,$results));
	}
    }

    # for masses > $mbreak we use a standard log grid
    {
	printf "Add masses > mbreak=%g\n",$mbreak if($vb);
	my $lnmmax=log($$opts{mmax});
	my $lnmmin=log($mbreak);
	my $nm=20.0;
	my $dlnm=($lnmmax-$lnmmin)/$nm;
	$lnmmin+=0.5*$dlnm;
	$lnmmax-=0.5*$dlnm;
	if($dlnm>0.0)
	{
	    for(my $lnm=$lnmmin;$lnm<=$lnmmax;$lnm+=$dlnm)
	    {
		push(@masses,exp($lnm));
	    }
	}
    }

    # trim masses outside grid range
    print "Trim masses outside range $$opts{mmin} and $$opts{mmax} : ";
    @masses = grep {
        $_ >= $$opts{mmin} && $_ <= $$opts{mmax}
    }@masses;
    print "Now have ",scalar @masses," masses\n";
    
    # add end points
    print "Add high end mass point $$opts{mmax} ... \n"if($vb);
    push(@masses,$$opts{mmin},$$opts{mmax});

    # make sure the masses are unique, sorted,
    # >=mmin and <=mmax and at spacing of no more than max_delta_m
    {
	printf "Check mass bounds are ok : %g >= M >= %g\n", $$opts{mmin}, $$opts{mmax}if($vb);
	my %m;
	map
	{
	    $m{$_}=1 if(($_ >= $$opts{mmin})&& ($_ <= $$opts{mmax}));
	}@masses;
	@masses=nsort(keys %m);
    
	if(defined($$opts{max_delta_m}))
	{
	    print "Check dm > $$opts{max_delta_m} for all masses\n"if($vb);
	    %m=(); # clear hash
	    my $n=$#masses;
	    $m{$masses[0]}=1;
	    for(my $i=1;$i<=$n;$i++)
	    {
		$m{$masses[$i]}=1;
		my $dm=$masses[$i]-$masses[$i-1];
		if($dm > $$opts{max_delta_m})
		{
		    my $dmstep=$dm/(1+int($dm/$$opts{max_delta_m}));
		    my $newm=$masses[$i-1];
		    while($newm<$masses[$i]-1e-5)
		    {
			$m{$newm}=1;
			print "Add extra mass (DM=$dm too large) at $newm\n"if($vb);
			$newm+=$dmstep;
		    }
		}
	    }
	    @masses=nsort(keys %m);
	}
    }
  
    print "Convert masslist to masshash\n"if($vb);
    $$results{mass_hash}=masslist_to_masshash(\@masses,$opts,$results);

    # and save the masses array
    @{$$results{mass_array}} = @masses;
    
    ### debugging
    if($$opts{debugging_output_directory})
    {
	print "Dump masslist to $$opts{debugging_output_directory}/adapt.masslist ... \n"if($vb);
	mkdirhier($$opts{debugging_output_directory});

	open(ADAPT_LIST,">$$opts{debugging_output_directory}/adapt.masslist");
	
	# show some options information
	print ADAPT_LIST "# options : \n#\n";
	map
	{
	    printf ADAPT_LIST "# opt %s = %s\n",$_,$$opts{$_} if(defined $_ && defined $$opts{$_});
	}keys %$opts;
	print "#\n";
	
	printf ADAPT_LIST "# %d stars in the masslist:\n",$#{$$results{mass_array}}+1;
	if($$opts{savegrid})
	{
	    open(ADAPT_LIST_DAT,">$$opts{debugging_output_directory}/adapt.masslist.dat"); 
	    printf ADAPT_LIST_DAT "# %d stars in the masslist:\n",$#{$$results{mass_array}}+1;
	    printf ADAPT_LIST_DAT "# %s %s %s %s %s %s %s %s\n",
	    'logM','M','t','log10+6','dt','dlogt','Mlow','Mhigh';
	}

	my $prevt=1e6;
	foreach my $m (@{$$results{mass_array}})
	{
	    my $dlogm=$$results{mass_hash}{$m};
	    my $t=stellar_lifetime($m,$results);
	    my $Dt=$prevt-$t;
	    my $Dlogt=log10($prevt)-log10($t);
	    printf ADAPT_LIST "logm=%g dlogm=%g m=%g t=%g log10t=%g (Dt=%g Dlogt=%g) \n",log($m),$$results{mass_hash}{$m},$m,$t,log10($t)+6.0,$Dt,$Dlogt;
	    if($$opts{savegrid})
	    {
		printf ADAPT_LIST_DAT "%g %g %g %g %g %g %g %g\n",log($m),$m,$t,log10($t)+6.0,$Dt,$Dlogt,$$results{mlow}{$m},$$results{mhigh}{$m};
	    }
	    $prevt=$t;
	}
	close ADAPT_LIST;
	close ADAPT_LIST_DAT if($$opts{savegrid});
	print "Masslist dumped to $$opts{debugging_output_directory}/adapt.masslist ... \n"if($vb);
    }

    print "MASSES @masses\n"if($vb);
    print "Made list containing ",scalar @masses," masses from $masses[0] to $masses[$#masses]\n";
}

sub stellar_lifetime
{
    my ($m,$results) = @_;
    my $lifetime = 10.0 ** (
        generic_interpolation_wrapper(
            log10($m),
            $$results{evoltimes_list},
            $$results{evoltimes_hash}
        ));
    print "Stellar lifetime = $lifetime from log10($m) (results=$results,",$$results{evoltimes_list},',',$$results{evoltimes_hash},")\n"if($vb);
    return $lifetime;
}

sub stellar_mass_from_lifetime
{    
    my $lifetime = $_[0];
    my $results = $_[1];
    my $m = 10.0 ** (
        generic_interpolation_wrapper(
            log10($_[0]),
            $$results{mtimes_list},
            $$results{mtimes_hash}
        ));
    print "Stellar mass = $m from lifetime = $lifetime (results=$results)\n"if($vb);
    return $m;
}

sub make_stellar_lifetime_cache
{
    # make the stellar lifetime cache directory
    my $opts = $_[0]; # options hash
    if(defined $$opts{stellar_lifetime_cachedir} &&
       !-d $$opts{stellar_lifetime_cachedir})
    {
        return rob_misc::mkdirhier($$opts{stellar_lifetime_cachedir});
    }
    else
    {
        return undef;
    }
}

sub use_stellar_lifetime_cache
{
    # return 1 if we can to use the stellar lifetime cache
    # return 0 otherwise 
    my $opts = $_[0]; # options hash
    return (defined $$opts{stellar_lifetime_cachedir} &&
            -d $$opts{stellar_lifetime_cachedir}) ? 1 : 0;
}

sub check_cache_for_stellar_lifetime_table
{
    # check the cached directory for a stellar lifetime table
    #
    # If found, it's set in $results and return 1
    #
    # If not found (or not implemented) return 0
    
    my $opts = $_[0]; # options hash
    return 0 if(use_stellar_lifetime_cache($opts)==0);
    my $results=$_[1]; # results hash

    # .info file contains the opts hash in Data::Dumper form
    my @infofiles = grep {
        /\.info$/
        }listfiles($$opts{stellar_lifetime_cachedir});

    print "SLTCache: found info files @infofiles\n"if($$opts{vb});
        
    # convert $opts into a Data::Dumper object
    my $dd = stellar_lifetime_opts_Data_Dumper_object($opts);
    my $dd_string = $dd->Dump;
    clean_ddstring(\$dd_string);
    
    foreach my $infofile (@infofiles)
    {
        print "SLTCache: Checking info file $infofile\n" if($$opts{vb});
        my $file_dd_string = rob_misc::slurp($$opts{stellar_lifetime_cachedir}.'/'.$infofile);
        clean_ddstring(\$file_dd_string);
        
        if($dd_string eq $file_dd_string)
        {
            print "SLTCache: Matched current run : loading and evaling into $results\n"if($$opts{vb});
            my $file = $infofile;
            $file=~s/\.info$//;
            my $results_dd = rob_misc::slurp($$opts{stellar_lifetime_cachedir}.'/'.$file);
            my $VAR1;
            eval $results_dd;
            %{$results} = %{$VAR1};

            # check hash and list sizes to see if the data was saved
            my $hashsize = defined $results->{evoltimes_hash} ?
                keys %{$results->{evoltimes_hash}} : 0;
            my $listsize = defined $results->{evoltimes_list} ?
                @{$results->{evoltimes_list}} : 0;
            
            if(!defined $results ||
               $hashsize<=0 ||
               $listsize<=0)
            {
                print "SLTCache: Stellar lifetimes table loaded from $file (info at $infofile) natches our run but failed to eval : is it corrupt? It has hash size $hashsize, list size $listsize\n";
                exit;
            }

            return 1;
        }
    }

    return 0;
}

sub clean_ddstring
{
    # given a set of options in a Data::Dumper string
    # clean out options that don't change the data
    my $dd = shift; # reference to dd string
    chomp $$dd;
    $$dd=~s/'vb' => \d+//; # vb doesn't matter
    $$dd=~s/'nthreads' => \d+//; # nthreads doesn't matter
    $$dd=~s/'time_adaptive_mass_grid_log10_time' => \d//; # doesn't matter
}

sub stellar_lifetime_opts_Data_Dumper_object
{
    # convert $opts to a Data::Dumper object
    my $opts = $_[0];
    my $dd = Data::Dumper->new([$opts]);
    $dd->Sortkeys(1);
    return $dd;
}

sub save_stellar_lifetime_table
{
    my $opts = $_[0]; # options hash
    return 0 if(use_stellar_lifetime_cache($opts)==0);
    my $results = $_[1]; # results hash
    make_stellar_lifetime_cache($opts);

    # make filename for cache file
    my $tmpnam = 
        $$opts{stellar_lifetime_cachedir}.'/'.
        File::Temp->new(
        TEMPLATE => 'XXXXXXXX'
        )->filename();

    my $file = $tmpnam;
    my $infofile = $tmpnam.'.info';
    
    print "SLTCache: Save stellar lifetime table to $file, info to $infofile\n" if($$opts{vb});

    # put info into data dumper object
    my $info = stellar_lifetime_opts_Data_Dumper_object($opts);

    # NB if output fails, just continue, do not die
    if(open(my $fp,'>',$infofile))
    {
        print {$fp} $info->Dump;
        close $fp;
        if(open(my $fp,'>',$file))
        {
            # put data into data dumper object
            my $dd = Data::Dumper->new([$results]);
            print {$fp} $dd->Dump;
            close $fp;
        }
        else
        {
            print STDERR "Write to $file failed\n";
        }
    }
    else
    {
        print STDERR "Write to $infofile failed\n";
    }
         
}

sub make_stellar_lifetime_table
{
    # make a table of stellar lifetimes : 
    # requires code build with SINGLE_STAR_LIFETIMES
    my $opts = $_[0]; # options hash
    my $results = $_[1]; # results hash

    # use min and max masses given by binary_c
    
    my $population = binary_grid2->new(
        
        # force single stars, long lived
        binary=>0,

        # disable counting of system errors
        add_up_system_errors=>0,
        
        # force modulo 1 and offset 0 so we run all the stars
        modulo=>1,
        offset=>0,

        # force single thread only
        nthreads=>1,
        
        # disable logging
        log_args=>0,
        log_filename=>'/dev/null',
        save_args=>0,
        
        return_array_refs => 1,

        # force evolution time and timestep
        max_evolution_time => 2.0*$$opts{max_evolution_time}//15000,

        timestep_modulator => 1.0,

        # do not allow evolution splitting on SN
        evolution_splitting => 0,

        # set up function to extract lifetimes
        parse_bse_function_pointer => \&parse_stellar_lifetime,

        # metallicity
        metallicity => $$opts{metallicity},

        # max evolution time
        max_evolution_time => $$opts{max_evolution_time} // 15000,
            
        # do not output yields unless we have to
        yields_dt => $$opts{yields_dt} // 1e8,

        vb=>$$opts{vb}//0,
        );

    # set results hash
    $population->{_results} = $results;

    # get the min and max stellar mass available to the evolution code
    # and use these to make the lifetimes table
    my $vstr = $population->evcode_version_string();
    my $lnmmax = log(($vstr=~/BINARY_C_MAXIMUM_STELLAR_MASS is (\S+)/)[0] // 100.0);
    my $lnmmin = log(($vstr=~/BINARY_C_MINIMUM_STELLAR_MASS is (\S+)/)[0] // 0.1);
    my $dlnm = ($lnmmax-$lnmmin)/$$opts{stellar_lifetime_table_nm}; # grid spacing

    printf "Making stellar lifetime table (nm=$$opts{stellar_lifetime_table_nm}) from M=%g Msun to %g Msun \n",
        exp($lnmmin),exp($lnmmax);

    ############################################################
    
    my @lnmasses;
    for(my $lnm=$lnmmin+0.5*$dlnm;$lnm<=$lnmmax;$lnm+=$dlnm)
    {    
	push(@lnmasses,$lnm);
    }
    
    # add extra around the helium flash
    if($$opts{extra_flash_resolution})
    {
	my $dm=0.2;
	my $mflash=mass_helium_flash($$opts{metallicity});
	my $lnmflash=log($mflash);
	my $lnmmin=log($mflash-$dm);
	my $lnmmax=log($mflash+$dm);
	my $dlnm=($lnmmax-$lnmmin)/100.0;
	for(my $lnm=$lnmmin;$lnm<=$lnmmax;$lnm+=$dlnm)
	{
	    push(@lnmasses,$lnm);
	}
    }

    # numerically sort
    @lnmasses = nsort @lnmasses;
    
    printf "Lifetimes table thread loop\n" if($vb);
    
    # make a queue of threads to carry out calculations
    print "Running models to make mass grid : please wait...\n";

    my $thread_q = Thread::Queue->new();
    
    # launch threads
    my @threads;
    foreach my $nthread (0..$$opts{nthreads}-1)
    {
        print "Launch thread $nthread\n";
	$threads[$nthread] =
            
	    threads->create(
                {context=>'list'},
                \&lifetime_thread,
                [$population,$nthread,$thread_q]
	    )||
	    die("could not create thread error in adpative_masshash");
    }

    # queue stars (no limit required, number is presumably small)
    my $nmasses = scalar @lnmasses; 
    while($#lnmasses>-1)
    {
	$thread_q->enqueue(shift @lnmasses);
    }

    # stop threads
    $thread_q->enqueue((undef) x ($$opts{nthreads}+1));

    my $npend = $thread_q->pending();
    my $npendwas = $npend; 
    my $sleepsecs = 1;

    while($npend>1)
    {
        printf "Stellar lifetime table done : %5.2f %%  ... %5.2f systems/s \x0d", 
            MAX(0.0,1.0-$npend/$nmasses) * 100.0,
            ($npendwas - $npend)/(1.0*$sleepsecs);
        $npendwas = $npend;
        sleep $sleepsecs;
        $npend = $thread_q->pending(); 
    }
    print "Done stellar lifetime table : joining threads\n";
        
    # join data
    foreach my $thread (@threads)
    {
	my $h = $thread->join;
	Hash::RobMerge::arithmetically_add_hashes($results,$h);
    }
   
    printf "Done models for mass grid, currently %d threads running\n",scalar(threads->list(threads::running));
    print "Finished threads\n"if($vb);
    
    @{$$results{evoltimes_list}}=nsort (keys %{$$results{evoltimes_hash}});
    @{$$results{mtimes_list}}=nsort(keys %{$$results{mtimes_hash}});

    if($vb)
    {
	foreach my $log10m (@{$$results{evoltimes_list}})
	{
	    printf "MT %g %g\n",$log10m,$$results{evoltimes_hash}{$log10m};
	}
    }

    if($$opts{debugging_output_directory})
    {
	open(TFP,">$$opts{debugging_output_directory}/evoltimes");
	foreach my $log10m (@{$$results{evoltimes_list}})
	{
	    printf TFP "%g %g\n",$log10m,$$results{evoltimes_hash}{$log10m};
	}
	close TFP;
    }

    print "Save table in cache\n";
    save_stellar_lifetime_table($opts,$results);
}

sub parse_stellar_lifetime
{
    my $population = shift;

    # cut-down parser
    my $m;
    my $t;
    my $h=$_[0]; # data hash (to be filled)

    while(1)
    {
	my $l = $population->tbse_line();
        my $header = shift @$l;
        last if ($header eq 'fin');
        if($header eq 'SINGLE_STAR_LIFETIME')
        {	
            $m = shift @$l;
            $t = shift @$l;
            $t *= 1.0;
        }
    }

    # if no time available, assume max evolution time
    if(defined($t))
    {
	$$h{evoltimes_hash}{log10($m)}=log10($t);
	$$h{mtimes_hash}{log10($t)}=log10($m);
    }
}

sub lifetime_thread
{
    my ($population,$nthread,$thread_q) = @{$_[0]};
    my %threadinfo = (
        thread_number => $nthread,
        evcode_pid => undef
        ); 
    
    my $pid = $population->tbse_launch(\%threadinfo);
    my @defaults = (
        M_2=>0.01,
        metallicity=>$population->{_bse_options}{metallicity},
        orbital_period=>$population->{_grid_options}{single_star_period},
        eccentricity=>0.0,
        probability=>1.0,
        phasevol=>1.0
        );
    my $vb = ($population->{_grid_options}->{vb}>=2);
    my $default_args = $population->make_evcode_arghash
            ({
                M_1=>1.0,
                @defaults,
             });
    my $h = {};
    while(my $lnm = $thread_q->dequeue())
    {
        my $args = {
            %$default_args,'M_1'=>exp($lnm)
        };
        if($vb)
        {
            printf "RUN M=%g (%s)\n",
                exp($lnm),$population->make_argstring($args) 
	}
        $population->tbse($args,$h);
    }
    print "lifetime_thread: $nthread : calling tbse_land on $nthread\n";
    $population->tbse_land($nthread,$pid);
    $population->tbse_kill();
    return $h;
}

sub masslist_to_masshash
{
    # given a list of (linear) masses make a (log) grid of them
    # with the appropriate dlnm weightings

    # output:
    my %masshash; # returned

    # input:
    my $masses=$_[0]; # pointer to the mass list
    my $opts=$_[1]; # hash of options
    my $results=$_[2]; # results hash
    
    # counter
    my $n=$#{$masses};

    # set up the masshash (the dlnm)
    my $lnmlow=log($$opts{mmin});
    my $lnmhigh=log($$opts{mmax});


    if($lnmlow == $lnmhigh)
    {
	my $m=$$masses[0];
        my $lnm = log($m);
	$masshash{$m} = 0.0; # assumed!
	if($$opts{savegrid})
	{
	    $$results{mlow}{$m}=$lnm;
	    $$results{mhigh}{$m}=$lnm;
	}
    }
    else
    {
        
        # first mass
        {
            print "Set up dlnm : ",exp($lnmlow),"\n"if($vb);
            my $m=$$masses[0];
            my $lnm=log($m);

            # take half the 'distance' to the next mass, 
            # and the distance from the lowest mass on the grid 
            # (given by lnmlow) to this mass
            $masshash{$m}=0.5*(log($$masses[1])+$lnm)-$lnmlow;
            if($$opts{savegrid})
            {
                $$results{mlow}{$m}=$lnmlow;
                $$results{mhigh}{$m}=$lnm+(log($$masses[1])-$lnm);
            }
            print "Set dlnm[0] at mass $$masses[0] = $masshash{$$masses[0]}\n"if($vb);
        }

        # generic point
        for(my $i=1;$i<$n;$i++)
        {
            my $m=$$masses[$i];
            my $lnm=log($m); 
            $masshash{$m}=0.5*(log($$masses[$i+1])-log($$masses[$i-1]));
            if($$opts{savegrid})
            {
                $$results{mlow}{$m}=$lnm-0.5*($lnm-log($$masses[$i-1]));
                $$results{mhigh}{$m}=$lnm+0.5*(log($$masses[$i+1])-$lnm);
            }
            print "Set dlnm[$i] at mass $$masses[$i] = $masshash{$$masses[$i]}\n"if($vb); 
        }

        # final point
        {
            my $m=$$masses[$n];
            my $lnm=log($m);
            $masshash{$m}=($lnmhigh-$lnm)+0.5*($lnm-log($$masses[$n-1]));
            
            if($$opts{savegrid})
            {
                $$results{mlow}{$m}=$lnm-0.5*($lnm-log($$masses[$n-1]));
                $$results{mhigh}{$m}=$lnmhigh;
            }
            print "Set dlnm[$n] at mass $$masses[$n] = $masshash{$$masses[$n]}\n"if($vb); 
        }
        
        # check we have filled the parameter space
        {
            # make global mlist
            my @mlist=nsort(values %masshash);
            my $dlnmsum=sum_array_with_pointers(\@mlist);
            if(abs($dlnmsum-(log($$opts{mmax})-log($$opts{mmin}))) / $dlnmsum > 1e-4)
            {
                printf STDERR "Grid lnm sum error: is %g should be %g\n",$dlnmsum,$lnmhigh-$lnmlow;
                $SIG{__DIE__} = undef;
                exit;
            }
        }
    }
    return \%masshash;
}

sub mass_helium_flash
{
    # maximum (initial) stellar mass for which the helium
    # flash occurs
    my $metallicity = $_[0] // 0.02;
    my $lzs=log10(50.0*$metallicity); # 50.0=1.0/0.02
    return 1.995 + $lzs*(0.25 + $lzs*0.087);
}



sub agbzoom
{
    # a constant distribution with a zoom region
    # designed for AGB stars

    # $opts is a hash pointer, passed in:
    # logperiod = the current period
    # logpermin to logpermax = period range 
    # dlogperiod = width of zoom range around logperiod
    # resolution = resolution when not zoomed
    # zoomfactor = zoom factor (<1)
    # m1,m2 = stellar masses
    # m1hbb only zoom if m1hbbmin < m1 < m1hbbmax
    my $opts=shift;

    # logging
    my $vb=0 || $$opts{vb};
    
    # find the period at which RLOF begins
    my $logagbper = log10(binary_grid2::agbperiod(undef,
                                                  $$opts{m1},
                                                  $$opts{m2},
                                                  $$opts{metallicity}));

    # hence the standard grid spacing
    my $C = ($$opts{logpermax}-$$opts{logpermin})/$$opts{resolution};

    # $d is the final returned spacing, default to the standard
    my $d = $C;

    # the zoomed spacing (which we might, or might not, use)
    my $dzoom = $d * $$opts{zoomfactor};
    
    # check if in required mass range
    my $m1_in_range = ($$opts{m1} > $$opts{m1hbbmin} && 
                       $$opts{m1} < $$opts{m1hbbmax}) ? 1 : 0;
    
    if($vb)
    {
        printf "M1=%g (in range? %d) M2=%g : logpermin=%g < [ %g < logP=%g (P=%g) < %g ] < logpermax=%g; res=%g zoomfactor=%g ",
        $$opts{m1},
        $m1_in_range,
        $$opts{m2},
        $$opts{logpermin},
        $logagbper-$$opts{dlogperiod}, 
        $$opts{logperiod},
        10.0**$$opts{logperiod},
        $logagbper+$$opts{dlogperiod}, 
        $$opts{logpermax},
        $$opts{resolution}, 
        $$opts{zoomfactor};
    }

    if($m1_in_range && 
       (abs($logagbper - $$opts{logperiod}) < $$opts{dlogperiod}))
    {
	# in zoom range : apply zoom
	$d = $dzoom;
	print "ZOOM! d=$d\n"if($vb);
    }
    else
    {
	# must not overshoot the zoom region 
	
	# lower edge of the zoom region
	my $lower = $logagbper - $$opts{dlogperiod};

	printf "nozoom! d=$d " if($vb);
 	if($m1_in_range &&
	   $$opts{logperiod} < $lower &&
	   $$opts{logperiod} + $d > $lower)
	{
	    $d = 1.00001 * ($lower - $$opts{logperiod});
	    print "revise near edge $d"if($vb);
	}
	print "\n"if($vb);
    }
    
    return $d;
}

# todo
# write a spacing function that takes equal steps in probability...
# (tricky!)


1;

__END__


